#!/usr/bin/env python3
"""
AWS Incident Manager Export Script - Simple Version
Exports incident data as structured JSON, incident by incident.
"""

import boto3
import json
import os
import sys
import re
from datetime import datetime, timezone
from typing import Dict, List, Any, Optional
import argparse
import logging

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('incident_manager_export.log'),
        logging.StreamHandler(sys.stdout)
    ]
)
logger = logging.getLogger(__name__)

# Global limits for API calls to prevent timeouts and excessive data
API_LIMITS = {
    'timeline_events': 100,         # Maximum timeline events to fetch per incident
    'timeline_event_details': 100,  # Maximum timeline event details to fetch
    'related_items': 50,            # Maximum related items to fetch per incident
    'engagements': 20,              # Maximum engagements to fetch per incident
    'automation_executions': 10,    # Maximum automation executions to process
    'findings': 50,                 # Maximum findings to fetch per incident
    'tags': 100,                    # Maximum tags to fetch per incident
    'post_incident_analysis': True, # Whether to fetch post incident analysis
    'analysis_documents': 50        # Maximum analysis documents to fetch
}

class IncidentManagerExporter:
    def __init__(self, region: str = 'us-east-1', profile: Optional[str] = None, api_limits: Optional[Dict[str, int]] = None):
        """Initialize the exporter with AWS session."""
        try:
            if profile:
                session = boto3.Session(profile_name=profile)
            else:
                session = boto3.Session()
            
            self.ssm_incidents = session.client('ssm-incidents', region_name=region)
            self.ssm = session.client('ssm', region_name=region)
            self.region = region
            self.export_timestamp = datetime.now(timezone.utc).strftime('%Y%m%d_%H%M%S')
            self.output_dir = f'incident_manager_export_{self.export_timestamp}'     
       
            # Use provided limits or defaults
            self.api_limits = api_limits or API_LIMITS.copy()
            
            # Create output directories with restricted permissions
            # Only the owner can access these directories
            os.makedirs(self.output_dir, mode=0o700, exist_ok=True)
            self.incidents_dir = os.path.join(self.output_dir, 'incident_records')
            self.analyses_dir = os.path.join(self.output_dir, 'post_incident_analyses')
            os.makedirs(self.incidents_dir, mode=0o700, exist_ok=True)
            os.makedirs(self.analyses_dir, mode=0o700, exist_ok=True)
            
            logger.info(f"Export directory created: {self.output_dir}")
            logger.info(f"Incident records will be saved to: {self.incidents_dir}")
            logger.info(f"Post incident analyses will be saved to: {self.analyses_dir}")
            logger.info(f"API limits: {self.api_limits}")
            
        except Exception as e:
            logger.error(f"Failed to initialize AWS clients: {e}")
            raise

    def export_all_incidents(self, limit: Optional[int] = None) -> None:
        """Export all incidents and post incident analyses in separate processes."""
        incident_count = 0  # Initialize at the start
        analysis_count = 0  # Initialize at the start
        
        try:
            logger.info("Starting export process...")
            
            # First loop: Export all incidents
            incident_count = self.export_incidents(limit)   
            
            # Second loop: Export all post incident analyses
            analysis_count = self.export_all_post_incident_analyses()
            
            logger.info(f"✅ Export completed: {incident_count} incidents and {analysis_count} analyses exported")
            logger.info(f"📁 Files saved to: {self.output_dir}")
            
        except Exception as e:
            logger.error(f"Export failed: {e}")
            logger.info(f"Partial results: {incident_count} incidents and {analysis_count} analyses exported")
            raise

    def export_incidents(self, limit: Optional[int] = None) -> int:
        """Export all incidents with complete details."""
        try:
            logger.info("Starting incident export...")
            
            incident_count = 0
            
            # Get incident list
            paginator = self.ssm_incidents.get_paginator('list_incident_records')
            
            for page in paginator.paginate():
                for incident_summary in page.get('incidentRecordSummaries', []):
                    incident_arn = None
                    try:
                        incident_arn = incident_summary['arn']
                        incident_id = incident_arn.split('/')[-1]
                        
                        logger.info(f"Processing incident {incident_count + 1}: {incident_summary.get('title', incident_arn)}")
                        
                        # Follow the specified workflow for each incident
                        incident_data = self.export_single_incident(incident_arn, incident_summary)
                        
                        if incident_data:
                            incident_count += 1
                            
                            # Create filename from title and creation time
                            incident_filename = self.create_incident_filename(incident_summary)
                            incident_file = os.path.join(self.incidents_dir, incident_filename)
                            
                            with open(incident_file, 'w') as f:
                                json.dump(incident_data, f, indent=2, default=str)
                            
                            logger.info(f"✅ Exported incident {incident_count}: {incident_summary.get('title', 'Unknown')} -> {incident_filename}")
                            
                        else:
                            logger.warning(f"❌ Failed to export incident: {incident_summary.get('title', incident_arn)}")
                        
                        # Check limit
                        if limit and incident_count >= limit:
                            logger.info(f"Reached limit of {limit} incidents, stopping export")
                            break
                        
                    except Exception as e:
                        logger.error(f"Failed to export incident {incident_arn or 'unknown'}: {e}")
                
                # Check limit for outer loop too
                if limit and incident_count >= limit:
                    break
            
            logger.info(f"✅ Incident export completed: {incident_count} incidents exported")
            return incident_count
            
        except Exception as e:
            logger.error(f"Incident export failed: {e}")
            raise

    def export_all_post_incident_analyses(self) -> int:
        """Export all post incident analysis documents using SSM APIs."""
        try:
            logger.info("Starting post incident analysis export...")
            
            analysis_count = 0
            max_docs = min(self.api_limits.get('analysis_documents', 50), 50)
            
            # List SSM documents of type ProblemAnalysis
            try:
                response = self.ssm.list_documents(
                    Filters=[
                        {
                            'Key': 'DocumentType',
                            'Values': ['ProblemAnalysis']
                        },
                        {
                            'Key': 'Owner',
                            'Values': ['Self']
                        }
                    ],
                    MaxResults=max_docs
                )
                
                documents = response.get('DocumentIdentifiers', [])
                logger.info(f"Found {len(documents)} ProblemAnalysis documents")
                
                if not documents:
                    logger.info("No post incident analysis documents found")
                    return 0
                
                # For each document, get the full details
                for i, doc in enumerate(documents):
                    try:
                        doc_name = doc.get('Name')
                        if not doc_name:
                            continue
                            
                        logger.info(f"Processing analysis document {i + 1}/{len(documents)}: {doc_name}")
                        
                        # Get the full document
                        doc_response = self.ssm.get_document(
                            Name=doc_name,
                            DocumentFormat='JSON',
                            DocumentVersion='$LATEST'
                        )
                        
                        # Create analysis filename based on document metadata
                        analysis_filename = self.create_analysis_filename(doc, doc_response)
                        analysis_file = os.path.join(self.analyses_dir, analysis_filename)
                        
                        # Prepare the analysis data
                        analysis_data = {
                            'document_metadata': doc,
                            'document_details': {
                                'Name': doc_response.get('Name'),
                                'DocumentVersion': doc_response.get('DocumentVersion'),
                                'DocumentFormat': doc_response.get('DocumentFormat'),
                                'DocumentType': doc_response.get('DocumentType'),
                                'Content': doc_response.get('Content'),
                                'CreatedDate': doc_response.get('CreatedDate'),
                                'DisplayName': doc_response.get('DisplayName'),
                                'ReviewStatus': doc_response.get('ReviewStatus'),
                                'Status': doc_response.get('Status'),
                                'StatusInformation': doc_response.get('StatusInformation'),
                                'VersionName': doc_response.get('VersionName'),
                                'AttachmentsContent': doc_response.get('AttachmentsContent', []),
                                'Requires': doc_response.get('Requires', [])
                            },
                            'export_metadata': {
                                'exported_at': datetime.now(timezone.utc).isoformat(),
                                'region': self.region,
                                'document_name': doc_name
                            }
                        }
                        
                        # Save the analysis document
                        with open(analysis_file, 'w') as f:
                            json.dump(analysis_data, f, indent=2, default=str)
                        
                        analysis_count += 1
                        logger.info(f"✅ Exported analysis {analysis_count}: {doc_name} -> {analysis_filename}")
                        
                    except Exception as doc_error:
                        logger.error(f"Failed to export analysis document {doc_name}: {doc_error}")
                        continue
                
            except Exception as list_error:
                logger.error(f"Failed to list ProblemAnalysis documents: {list_error}")
                return 0
            
            logger.info(f"✅ Post incident analysis export completed: {analysis_count} analyses exported")
            return analysis_count
                
        except Exception as e:
            logger.error(f"Post incident analysis export failed: {e}")
            return 0

    def export_single_incident(self, incident_arn: str, incident_summary: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Export a single incident following the exact workflow specified."""
        try:
            logger.debug(f"Step 1: Getting incident details for {incident_arn}")
            # Step 1: Get incident details (get-incident-record)
            detail_response = self.ssm_incidents.get_incident_record(arn=incident_arn)
            incident_record = detail_response['incidentRecord']
            
            logger.debug(f"Step 2: Getting timeline events for {incident_arn}")
            # Step 2: List all timeline events (list-timeline-events)
            timeline_events = self.get_timeline_events(incident_arn)
            
            logger.debug(f"Step 3: Getting related items for {incident_arn}")
            # Step 3: List all related items (list-related-items)
            related_items = self.get_related_items(incident_arn)
            
            logger.debug(f"Step 4: Getting engagements for {incident_arn}")
            # Step 4: List all engagements (list-engagements)
            engagements = self.get_engagements(incident_arn)
            
            logger.debug(f"Step 5: Getting automation executions for {incident_arn}")
            # Step 5: List all automation executions (from incident record)
            automation_executions = self.get_automation_executions(incident_record)
            
            logger.debug(f"Step 6: Getting findings for {incident_arn}")
            # Step 6: List all findings (list-findings)
            findings = self.get_findings(incident_arn)
            
            logger.debug(f"Step 7: Getting tags for {incident_arn}")
            # Step 7: List all tags (list-tags-for-resource)
            tags = self.get_tags(incident_arn)
            
            # Step 8: Extract and enhance incident source details
            logger.debug(f"Step 8: Processing incident source details for {incident_arn}")
            incident_source_details = self.get_incident_source_details(incident_record, incident_summary)
            
            # Step 9: Get post incident analysis
            logger.debug(f"Step 9: Getting post incident analysis for {incident_arn}")
            post_incident_analysis = self.get_post_incident_analysis(incident_arn)
            
            # Compile complete incident data
            incident_data = {
                'incident_record': incident_record,
                'incident_summary': incident_summary,
                'incident_source_details': incident_source_details,
                'timeline_events': timeline_events,
                'related_items': related_items,
                'engagements': engagements,
                'automation_executions': automation_executions,
                'findings': findings,
                'tags': tags,
                'post_incident_analysis': post_incident_analysis,
                'export_metadata': {
                    'exported_at': datetime.now(timezone.utc).isoformat(),
                    'region': self.region,
                    'incident_arn': incident_arn
                }
            }
            
            return incident_data
            
        except Exception as e:
            logger.error(f"Failed to export incident {incident_arn}: {e}")
            return None

    def get_timeline_events(self, incident_arn: str) -> Dict[str, Any]:
        """Get timeline events for an incident with detailed information, respecting limits."""
        try:
            logger.debug(f"Getting timeline events for {incident_arn}")
            timeline_event_summaries = []
            paginator = self.ssm_incidents.get_paginator('list_timeline_events')
            
            page_count = 0
            total_events = 0
            max_events = self.api_limits['timeline_events']
            
            for page in paginator.paginate(incidentRecordArn=incident_arn):
                page_count += 1
                logger.debug(f"Processing timeline events page {page_count}")
                page_events = page.get('eventSummaries', [])
                
                # Check if we would exceed the limit
                if total_events + len(page_events) > max_events:
                    # Take only what we need to reach the limit
                    remaining = max_events - total_events
                    timeline_event_summaries.extend(page_events[:remaining])
                    total_events = max_events
                    logger.info(f"Reached timeline events limit of {max_events} for {incident_arn}")
                    break
                else:
                    timeline_event_summaries.extend(page_events)
                    total_events += len(page_events)
            
            logger.debug(f"Found {len(timeline_event_summaries)} timeline event summaries (limited to {max_events})")
            
            # Get detailed information for timeline events (with separate limit)
            detailed_timeline_events = []
            max_details = self.api_limits['timeline_event_details']
            events_to_detail = timeline_event_summaries[:max_details]
            
            for i, event_summary in enumerate(events_to_detail):
                try:
                    event_id = event_summary.get('eventId')
                    if event_id:
                        logger.debug(f"Getting detailed timeline event {i+1}/{len(events_to_detail)}: {event_id}")
                        detail_response = self.ssm_incidents.get_timeline_event(
                            eventId=event_id,
                            incidentRecordArn=incident_arn
                        )
                        # Combine summary and detailed information
                        combined_event = {
                            'summary': event_summary,
                            'details': detail_response.get('event', {})
                        }
                        detailed_timeline_events.append(combined_event)
                    else:
                        # If no event ID, just include the summary
                        detailed_timeline_events.append({
                            'summary': event_summary,
                            'details': None
                        })
                        
                except Exception as e:
                    logger.debug(f"Could not get timeline event details for {event_id}: {e}")
                    # Include summary even if details fail
                    detailed_timeline_events.append({
                        'summary': event_summary,
                        'details': None,
                        'error': str(e)
                    })
            
            # Include remaining summaries without details if we hit the detail limit
            remaining_summaries = []
            if len(timeline_event_summaries) > max_details:
                for event_summary in timeline_event_summaries[max_details:]:
                    remaining_summaries.append({
                        'summary': event_summary,
                        'details': None,
                        'note': 'Details not fetched due to limit'
                    })
            
            result = {
                'detailed_events': detailed_timeline_events,
                'summary_only_events': remaining_summaries,
                'metadata': {
                    'total_events_found': total_events,
                    'events_with_details': len(detailed_timeline_events),
                    'events_summary_only': len(remaining_summaries),
                    'limits_applied': {
                        'max_events': max_events,
                        'max_details': max_details
                    },
                    'truncated': total_events >= max_events
                }
            }
            
            logger.debug(f"Retrieved {len(detailed_timeline_events)} detailed + {len(remaining_summaries)} summary-only timeline events")
            return result
            
        except Exception as e:
            logger.error(f"Failed to get timeline events for {incident_arn}: {e}")
            return {
                'detailed_events': [],
                'summary_only_events': [],
                'metadata': {'error': str(e)},
                'error': str(e)
            }

    def get_related_items(self, incident_arn: str) -> Dict[str, Any]:
        """Get related items for an incident, respecting limits."""
        try:
            logger.debug(f"Getting related items for {incident_arn}")
            related_items = []
            paginator = self.ssm_incidents.get_paginator('list_related_items')
            
            page_count = 0
            total_items = 0
            max_items = self.api_limits['related_items']
            
            for page in paginator.paginate(incidentRecordArn=incident_arn):
                page_count += 1
                logger.debug(f"Processing related items page {page_count}")
                page_items = page.get('relatedItems', [])
                
                # Check if we would exceed the limit
                if total_items + len(page_items) > max_items:
                    # Take only what we need to reach the limit
                    remaining = max_items - total_items
                    related_items.extend(page_items[:remaining])
                    total_items = max_items
                    logger.info(f"Reached related items limit of {max_items} for {incident_arn}")
                    break
                else:
                    related_items.extend(page_items)
                    total_items += len(page_items)
            
            result = {
                'items': related_items,
                'metadata': {
                    'total_items': len(related_items),
                    'limit_applied': max_items,
                    'truncated': total_items >= max_items
                }
            }
            
            logger.debug(f"Found {len(related_items)} related items (limited to {max_items})")
            return result
            
        except Exception as e:
            logger.error(f"Failed to get related items for {incident_arn}: {e}")
            return {
                'items': [],
                'metadata': {'error': str(e)},
                'error': str(e)
            }

    def get_engagements(self, incident_arn: str) -> Dict[str, Any]:
        """Get engagements for an incident, respecting limits."""
        try:
            engagements = []
            max_engagements = self.api_limits['engagements']
            
            try:
                response = self.ssm_incidents.list_engagements(
                    incidentRecordArn=incident_arn
                )
                all_engagements = response.get('engagements', [])
                
                # Apply limit
                engagements = all_engagements[:max_engagements]
                if len(all_engagements) > max_engagements:
                    logger.info(f"Limited engagements to {max_engagements} out of {len(all_engagements)} for {incident_arn}")
                
                # Get detailed engagement information
                detailed_engagements = []
                for i, engagement in enumerate(engagements):
                    try:
                        engagement_arn = engagement.get('engagementArn')
                        if engagement_arn:
                            logger.debug(f"Getting engagement details {i+1}/{len(engagements)}")
                            detail_response = self.ssm_incidents.get_engagement(
                                engagementArn=engagement_arn
                            )
                            detailed_engagements.append(detail_response)
                        else:
                            detailed_engagements.append(engagement)
                    except Exception as e:
                        logger.debug(f"Could not get engagement details: {e}")
                        detailed_engagements.append(engagement)
                
                result = {
                    'engagements': detailed_engagements,
                    'metadata': {
                        'total_engagements': len(detailed_engagements),
                        'total_found': len(all_engagements),
                        'limit_applied': max_engagements,
                        'truncated': len(all_engagements) > max_engagements
                    }
                }
                
                return result
                
            except Exception as e:
                logger.debug(f"Could not retrieve engagements for {incident_arn}: {e}")
                return {
                    'engagements': [],
                    'metadata': {'error': str(e)},
                    'error': str(e)
                }
            
        except Exception as e:
            logger.error(f"Failed to get engagements for {incident_arn}: {e}")
            return {
                'engagements': [],
                'metadata': {'error': str(e)},
                'error': str(e)
            }    
            
    def get_automation_executions(self, incident_record: Dict[str, Any]) -> List[Dict[str, Any]]:
        """Get automation executions from incident record."""
        try:
            automation_executions = []
            
            # Get automation executions from incident record
            automations = incident_record.get('automationExecutions', [])
            
            # Get detailed information for each automation execution
            for automation in automations:
                try:
                    execution_id = automation.get('ssmAutomationExecutionArn')
                    if execution_id:
                        # Extract execution ID from ARN
                        execution_id = execution_id.split('/')[-1]
                        response = self.ssm.get_automation_execution(
                            AutomationExecutionId=execution_id
                        )
                        automation_executions.append(response.get('AutomationExecution', {}))
                    else:
                        automation_executions.append(automation)
                        
                except Exception as e:
                    logger.debug(f"Could not get automation execution details: {e}")
                    automation_executions.append(automation)
            
            return automation_executions
            
        except Exception as e:
            logger.error(f"Failed to get automation executions: {e}")
            return []

    def get_findings(self, incident_arn: str) -> List[Dict[str, Any]]:
        """Get findings for an incident."""
        try:
            findings = []
            
            try:
                response = self.ssm_incidents.batch_get_incident_findings(
                    incidentRecordArn=incident_arn
                )
                findings = response.get('findings', [])
            except Exception as e:
                logger.debug(f"No findings available for {incident_arn}: {e}")
            
            return findings
            
        except Exception as e:
            logger.error(f"Failed to get findings for {incident_arn}: {e}")
            return []

    def get_tags(self, incident_arn: str) -> List[Dict[str, Any]]:
        """Get tags for an incident."""
        try:
            tags = []
            
            try:
                response = self.ssm_incidents.list_tags_for_resource(
                    resourceArn=incident_arn
                )
                tags = response.get('tags', [])
                
            except Exception as e:
                logger.debug(f"Could not retrieve tags for {incident_arn}: {e}")
            
            return tags
            
        except Exception as e:
            logger.error(f"Failed to get tags for {incident_arn}: {e}")
            return []    
    
    def get_incident_source_details(self, incident_record: Dict[str, Any], incident_summary: Dict[str, Any]) -> Dict[str, Any]:
        """Extract and enhance incident source details from both record and summary."""
        try:
            # Get source details from incident record
            record_source = incident_record.get('incidentRecordSource', {})
            
            # Get source details from incident summary (may have additional info)
            summary_source = incident_summary.get('incidentRecordSource', {})
            
            # Combine and enhance the source information
            source_details = {
                'from_incident_record': record_source,
                'from_incident_summary': summary_source,
                'enhanced_details': {
                    'created_by': record_source.get('createdBy') or summary_source.get('createdBy'),
                    'source': record_source.get('source') or summary_source.get('source'),
                    'invoked_by': record_source.get('invokedBy') or summary_source.get('invokedBy'),
                    'resource_arn': record_source.get('resourceArn') or summary_source.get('resourceArn'),
                    'source_analysis': self._analyze_incident_source(record_source, summary_source)
                }
            }
            
            return source_details
            
        except Exception as e:
            logger.error(f"Failed to process incident source details: {e}")
            return {
                'from_incident_record': {},
                'from_incident_summary': {},
                'enhanced_details': {},
                'error': str(e)
            }

    def _analyze_incident_source(self, record_source: Dict[str, Any], summary_source: Dict[str, Any]) -> Dict[str, Any]:
        """Analyze and categorize the incident source information."""
        try:
            source = record_source.get('source') or summary_source.get('source', '')
            created_by = record_source.get('createdBy') or summary_source.get('createdBy', '')
            invoked_by = record_source.get('invokedBy') or summary_source.get('invokedBy')
            resource_arn = record_source.get('resourceArn') or summary_source.get('resourceArn')
            
            analysis = {
                'source_type': 'unknown',
                'creation_method': 'unknown',
                'automation_involved': False,
                'aws_service_involved': None,
                'human_created': False
            }
            
            # Analyze source type
            if source == 'aws.ssm-incidents.custom':
                analysis['source_type'] = 'manual'
                analysis['creation_method'] = 'manual_creation'
                analysis['human_created'] = True
            elif source == 'aws.securityhub':
                analysis['source_type'] = 'security_hub'
                analysis['creation_method'] = 'automated_security_finding'
                analysis['automation_involved'] = True
                analysis['aws_service_involved'] = 'SecurityHub'
            elif source == 'aws.cloudwatch':
                analysis['source_type'] = 'cloudwatch'
                analysis['creation_method'] = 'cloudwatch_alarm'
                analysis['automation_involved'] = True
                analysis['aws_service_involved'] = 'CloudWatch'
            elif 'events.amazonaws.com' in str(invoked_by):
                analysis['source_type'] = 'eventbridge'
                analysis['creation_method'] = 'eventbridge_rule'
                analysis['automation_involved'] = True
                analysis['aws_service_involved'] = 'EventBridge'
            
            # Analyze created by
            if 'assumed-role' in created_by:
                if 'Amazon_EventBridge_Invoke_SsmIncidents' in created_by:
                    analysis['creation_method'] = 'eventbridge_automated'
                    analysis['automation_involved'] = True
                elif 'Isengard' in created_by:
                    analysis['human_created'] = True
                    analysis['creation_method'] = 'human_via_console'
            
            # Extract service from resource ARN if available
            if resource_arn:
                try:
                    arn_parts = resource_arn.split(':')
                    if len(arn_parts) >= 3:
                        analysis['resource_service'] = arn_parts[2]
                except:
                    pass
            
            return analysis
            
        except Exception as e:
            logger.debug(f"Failed to analyze incident source: {e}")
            return {'error': str(e)}    
            
    def create_incident_filename(self, incident_summary: Dict[str, Any]) -> str:
        """Create a filename from incident title and creation time."""
        try:
            # Get title and clean it for filename
            title = incident_summary.get('title', 'Unknown_Incident')
            
            # Clean title for filename (remove/replace invalid characters)
            # Security: Remove path traversal, and leading special characters
            clean_title = re.sub(r'[<>:"/\\|?*\.]', '_', title)  # Replace filesystem-unsafe chars including dots
            clean_title = re.sub(r'\.\.', '_', clean_title)      # Prevent path traversal attacks (../)
            clean_title = re.sub(r'^[._-]+', '', clean_title)    # Remove leading special chars to prevent hidden files
            clean_title = re.sub(r'\s+', '_', clean_title)       # Replace spaces with underscores
            clean_title = clean_title.strip('_')                 # Clean up trailing underscores
            
            # Security: Ensure filename doesn't start with special characters that could create hidden files
            if clean_title and clean_title[0] in '.-_':
                clean_title = 'file_' + clean_title
            
            # Limit title length to avoid filesystem issues
            if len(clean_title) > 50:
                clean_title = clean_title[:50].rstrip('_')
            
            # Get creation time
            creation_time = incident_summary.get('creationTime')
            if creation_time:
                if hasattr(creation_time, 'strftime'):
                    # It's a datetime object
                    time_str = creation_time.strftime('%Y%m%d_%H%M%S')
                else:
                    # It might be a string or timestamp
                    try:
                        if isinstance(creation_time, str):
                            # Try to parse ISO format
                            from dateutil import parser
                            dt = parser.parse(creation_time)
                            time_str = dt.strftime('%Y%m%d_%H%M%S')
                        else:
                            # Assume it's a timestamp
                            dt = datetime.fromtimestamp(creation_time, tz=timezone.utc)
                            time_str = dt.strftime('%Y%m%d_%H%M%S')
                    except:
                        time_str = 'unknown_time'
            else:
                time_str = 'unknown_time'
            
            # Create filename
            filename = f"{time_str}_{clean_title}.json"
            
            # Ensure filename isn't too long (most filesystems have 255 char limit)
            if len(filename) > 200:
                # Truncate title part if needed
                max_title_len = 200 - len(time_str) - 6  # 6 for underscore and .json
                clean_title = clean_title[:max_title_len].rstrip('_')
                filename = f"{time_str}_{clean_title}.json"
            
            return filename
            
        except Exception as e:
            logger.debug(f"Failed to create filename from incident summary: {e}")
            # Fallback to a simple filename
            return f"incident_{datetime.now(timezone.utc).strftime('%Y%m%d_%H%M%S_%f')}.json"

    def get_post_incident_analysis(self, incident_arn: str) -> Dict[str, Any]:
        """Get post incident analysis for an incident from incident record."""
        try:
            logger.debug(f"Getting post incident analysis reference for {incident_arn}")
            
            # Get basic analysis info from incident record
            analysis_data = {
                'analysis_reference': None,
                'metadata': {
                    'has_analysis_reference': False,
                    'error': None
                }
            }
            
            try:
                # Get analysis summary from incident record
                response = self.ssm_incidents.get_incident_record(arn=incident_arn)
                incident_record = response.get('incidentRecord', {})
                
                # Check if there's a post incident analysis reference
                if 'postIncidentAnalysisSummary' in incident_record:
                    analysis_summary = incident_record['postIncidentAnalysisSummary']
                    analysis_data['analysis_reference'] = analysis_summary
                    analysis_data['metadata']['has_analysis_reference'] = True
                    logger.debug(f"Found post incident analysis reference for {incident_arn}")
                else:
                    logger.debug(f"No post incident analysis reference found for {incident_arn}")
                    
            except Exception as e:
                logger.debug(f"Could not retrieve post incident analysis reference for {incident_arn}: {e}")
                analysis_data['metadata']['error'] = str(e)
            
            return analysis_data
            
        except Exception as e:
            logger.error(f"Failed to get post incident analysis reference for {incident_arn}: {e}")
            return {
                'analysis_reference': None,
                'metadata': {
                    'has_analysis_reference': False,
                    'error': str(e)
                }
            } 

    def create_analysis_filename(self, doc_metadata: Dict[str, Any], doc_details: Dict[str, Any]) -> str:
        """Create a filename for analysis document based on creation time and document name."""
        try:
            # Get document name and clean it for filename
            doc_name = doc_details.get('DisplayName') or doc_details.get('Name', 'Unknown_Analysis')
            
            # Clean name for filename (remove/replace invalid characters)
            # Security: Remove path traversal, and leading special characters
            clean_name = re.sub(r'[<>:"/\\|?*\.]', '_', doc_name)  # Replace filesystem-unsafe chars including dots
            clean_name = re.sub(r'\.\.', '_', clean_name)          # Prevent path traversal attacks (../)
            clean_name = re.sub(r'^[._-]+', '', clean_name)        # Remove leading special chars to prevent hidden files
            clean_name = re.sub(r'\s+', '_', clean_name)           # Replace spaces with underscores
            clean_name = clean_name.strip('_')                     # Clean up trailing underscores
            
            # Security: Ensure filename doesn't start with special characters that could create hidden files
            if clean_name and clean_name[0] in '.-_':
                clean_name = 'file_' + clean_name
            
            # Limit name length to avoid filesystem issues
            if len(clean_name) > 50:
                clean_name = clean_name[:50].rstrip('_')
            
            # Get creation time from document details or metadata
            creation_time = doc_details.get('CreatedDate') or doc_metadata.get('CreatedDate')
            if creation_time:
                try:
                    if isinstance(creation_time, (int, float)):
                        # It's a timestamp
                        dt = datetime.fromtimestamp(creation_time, tz=timezone.utc)
                        time_str = dt.strftime('%Y%m%d_%H%M%S')
                    elif hasattr(creation_time, 'strftime'):
                        # It's a datetime object
                        time_str = creation_time.strftime('%Y%m%d_%H%M%S')
                    else:
                        # Try to parse as string
                        from dateutil import parser
                        dt = parser.parse(str(creation_time))
                        time_str = dt.strftime('%Y%m%d_%H%M%S')
                except:
                    time_str = 'unknown_time'
            else:
                time_str = 'unknown_time'
            
            # Create filename
            filename = f"{time_str}_{clean_name}.json"
            
            # Ensure filename isn't too long (most filesystems have 255 char limit)
            if len(filename) > 200:
                # Truncate name part if needed
                max_name_len = 200 - len(time_str) - 6  # 6 for underscore and .json
                clean_name = clean_name[:max_name_len].rstrip('_')
                filename = f"{time_str}_{clean_name}.json"
            
            return filename
            
        except Exception as e:
            logger.debug(f"Failed to create analysis filename: {e}")
            # Fallback to a simple filename
            doc_id = doc_details.get('Name', 'unknown')[:20]  # Use first 20 chars of document name
            return f"analysis_{datetime.now(timezone.utc).strftime('%Y%m%d_%H%M%S')}_{doc_id}.json"


def main():
    """Main execution function."""
    parser = argparse.ArgumentParser(description='Export AWS Incident Manager resources')
    parser.add_argument('--region', default='us-east-1', help='AWS region (default: us-east-1)')
    parser.add_argument('--profile', help='AWS profile to use')
    parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose logging')
    parser.add_argument('--limit', type=int, help='Limit number of incidents to process (for testing)')
    
    # API limit arguments
    parser.add_argument('--timeline-events-limit', type=int, default=API_LIMITS['timeline_events'],
                       help=f'Max timeline events per incident (default: {API_LIMITS["timeline_events"]})')
    parser.add_argument('--timeline-details-limit', type=int, default=API_LIMITS['timeline_event_details'],
                       help=f'Max timeline event details per incident (default: {API_LIMITS["timeline_event_details"]})')
    parser.add_argument('--related-items-limit', type=int, default=API_LIMITS['related_items'],
                       help=f'Max related items per incident (default: {API_LIMITS["related_items"]})')
    parser.add_argument('--engagements-limit', type=int, default=API_LIMITS['engagements'],
                       help=f'Max engagements per incident (default: {API_LIMITS["engagements"]})')
    parser.add_argument('--analysis-docs-limit', type=int, default=API_LIMITS['analysis_documents'],
                       help=f'Max analysis documents to fetch (default: {API_LIMITS["analysis_documents"]})')
    
    args = parser.parse_args()
    
    if args.verbose:
        logging.getLogger().setLevel(logging.DEBUG)
    
    try:
        # Create custom API limits from arguments
        custom_limits = {
            'timeline_events': args.timeline_events_limit,
            'timeline_event_details': args.timeline_details_limit,
            'related_items': args.related_items_limit,
            'engagements': args.engagements_limit,
            'automation_executions': API_LIMITS['automation_executions'],
            'findings': API_LIMITS['findings'],
            'tags': API_LIMITS['tags'],
            'post_incident_analysis': API_LIMITS['post_incident_analysis'],
            'analysis_documents': args.analysis_docs_limit
        }
        
        # Initialize exporter
        exporter = IncidentManagerExporter(region=args.region, profile=args.profile, api_limits=custom_limits)
        
        # Run export
        exporter.export_all_incidents(limit=args.limit)
        
        print(f"\n✅ Export completed successfully!")
        print(f"📁 Output directory: {exporter.output_dir}")
        
    except Exception as e:
        logger.error(f"Export failed: {e}")
        sys.exit(1)


if __name__ == '__main__':
    main()